###
# Copyright 2016-2021 Hewlett Packard Enterprise, Inc. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#  http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
###

# -*- coding: utf-8 -*-
""" Server State Command for rdmc """

from rdmc_helper import (
    ReturnCodes,
    InvalidCommandLineError,
    Encryption,
    InvalidCommandLineErrorOPTS,
    NoContentsFoundForOperationError,
)


class ServerStateCommand:
    """Returns the current state of the server that  is currently logged in"""

    def __init__(self):
        self.ident = {
            "name": "serverstate",
            "usage": None,
            "description": "Returns the current state of the"
            " server\n\n\tExample: serverstate",
            "summary": "Returns the current state of the server.",
            "aliases": [],
            "auxcommands": [],
        }
        self.cmdbase = None
        self.rdmc = None
        self.auxcommands = dict()

    def run(self, line, help_disp=False):
        """Main serverstate function

        :param line: string of arguments passed in
        :type line: str.
        """
        if help_disp:
            self.parser.print_help()
            return ReturnCodes.SUCCESS
        try:
            (options, args) = self.rdmc.rdmc_parse_arglist(self, line)
        except (InvalidCommandLineErrorOPTS, SystemExit):
            if ("-h" in line) or ("--help" in line):
                return ReturnCodes.SUCCESS
            else:
                raise InvalidCommandLineErrorOPTS("")

        if args:
            raise InvalidCommandLineError(
                "Invalid number of parameters, "
                "serverstate command does not take any parameters."
            )

        self.serverstatevalidation(options)

        path = self.rdmc.app.typepath.defs.systempath
        results = self.rdmc.app.get_handler(path, silent=True, uncache=True).dict

        if results:
            results = results["Oem"][self.rdmc.app.typepath.defs.oemhp]["PostState"]
            self.rdmc.ui.printer("The server is currently in state: " + results + "\n")
        else:
            raise NoContentsFoundForOperationError("Unable to retrieve server state")

        self.cmdbase.logout_routine(self, options)
        # Return code
        return ReturnCodes.SUCCESS

    def serverstatevalidation(self, options):
        """Server state method validation function

        :param options: command line options
        :type options: list.
        """
        self.cmdbase.login_select_validation(self, options)

    def definearguments(self, customparser):
        """Wrapper function for new command main function

        :param customparser: command line input
        :type customparser: parser.
        """
        if not customparser:
            return

        self.cmdbase.add_login_arguments_group(customparser)
